using System;
using System.IO;
using System.Text;

namespace ECGMonitor
{
	#region Event definitions
	/// <summary>
	/// Arguments for ECG device event
	/// </summary>
    public class EKGReceiveDataEventArgs : EventArgs
    {
		/// <summary>Current states of the ECG device</summary>
		public readonly byte reportID;
        public readonly short[] data;
		/// <summary>
		/// Constructor
		/// </summary>
        public EKGReceiveDataEventArgs(byte id, short[] inputData)
        {
            reportID = id;
			data = inputData;
        }
    }
	/// <summary>
	/// Delegate for ECG device event
	/// </summary>
    public delegate void EKGReceiveDataEventHandler(object sender, EKGReceiveDataEventArgs args);
	#endregion

	/// <summary>
	/// Class that defines a controller of Buzz handsets : Representation of the USB hardware device
	/// </summary>
    public class ECGDevice : HIDDevice
    {
		#region Public attributes/methods
		/// <summary>
		/// Event fired when ECG state changes
		/// </summary>
		public event EKGReceiveDataEventHandler OnInputReportDataReceived;
		/// <summary>
		/// Creates an input report for use in the HID device framework
		/// </summary>
		/// <returns>A new input report for this device</returns>
		public override InputReport CreateInputReport()
		{
			return new ECGInputReport(this);
		}
		/// <summary>
		/// Finds the Buzz handset. 
		/// </summary>
		/// <returns>A new BuzzHandsetDevice or null if not found.</returns>
        public static ECGDevice FindECGDevice(int VID, int PID)
        {
            return (ECGDevice)FindDevice(VID, PID, typeof(ECGDevice));
        }
		#endregion

		#region Overrides
		/// <summary>
		/// Fired when data has been received over USB
		/// </summary>
		/// <param name="oInRep">Input report received</param>
		protected override void HandleDataReceived(InputReport oInRep)
		{
			// Fire the event handler if assigned
			if (OnInputReportDataReceived != null)
			{
				ECGInputReport oECGIn = (ECGInputReport)oInRep;
				OnInputReportDataReceived(this, new EKGReceiveDataEventArgs(oECGIn.ReportID, oECGIn.Data));
			}
		}
		/// <summary>
		/// Dispose.
		/// </summary>
		/// <param name="bDisposing">True if object is being disposed - else is being finalised</param>
		protected override void Dispose(bool bDisposing)
		{
			if ( bDisposing )
			{
			}
			base.Dispose(bDisposing);
		}
		#endregion
    }

	#region Device reports
	/// <summary>
	/// Output report for Buzz device
	/// </summary>
	public class ECGOutputReport : OutputReport
	{
		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="oDev">Device constructing this report</param>
		public ECGOutputReport(HIDDevice oDev) : base(oDev) {}

	}
	/// <summary>
	/// Input report for Buzz Handset device
	/// </summary>
	public class ECGInputReport : InputReport
	{
        private byte m_reportID;
        private short[] m_data;

        public byte ReportID
        {
            get { return m_reportID; }
            set { m_reportID = value; }
        }
	
        public short[] Data
        {
            get { return m_data; }
            set { m_data = value; }
        }
	
		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="oDev">Constructing device</param>
		public ECGInputReport(HIDDevice oDev) : base(oDev)
		{
            m_reportID = 0;
/*
            for (int i = 0; i < xv.Length; ++i)
            {
                xv[i] = 0.0f;
                yv[i] = 0.0f;
            }
 */
		}
        /// <summary>
        /// Process the raw data from the actual input report bytes
        /// </summary>
        public override void ProcessData()
		{
			byte[] arrData = Buffer;
            m_reportID = arrData[0];
            int size = (arrData.Length)/2;
            m_data = new short[size];

            int i = 1;
            int j = 0;
            while(i < arrData.Length)
            {
                m_data[j++] = (short)(arrData[i] + (arrData[i+1] << 8));
                i += 2;
            }
		}

	}
	#endregion
}
