using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.IO.Ports;
using System.IO;

namespace ECGMonitor
{
    public partial class ECGForm : Form
    {
        #region globals
        //RRS: Modified the VID and PID to test the Serial to USB connection.
        //      This was unsuccessful because the Serial to USB is installed and 
        //      recognized as a serial device (COM port) not as a HID.
        //      Therefore when the program searches through the all HID's the 
        //      serial to USB cable will not be present.
        const int STM32_CIRCLE_ECG_VID = /*0x0A5C;*/0x0000;
        const int STM32_CIRCLE_ECG_VIP = /*0x2101;*/0x0000;
        private double[] xv = new double[9];
        private double[] yv = new double[9];
        private ECGDevice m_oECGDevice = null;
                
        private double gain = 250;
        private bool gainChanged = false;

        //variables to calculate heart rate
        private List<float> dataBuffer = new List<float>();
        private int prevPt = 0;
        private int currentPt = 0;        
        private int threshold = 2400;
        private bool trigger = false;
        private int rWave = 0;
        private double startTime = 0;
        private double endTime = 0;
        private double frequency = 0;
        private double period = 0;
        private double heartRate = 0;
        private double averageHeartRate = 0;
        private Queue hrQ = new Queue();        
        
        //variables to store GPS data 
        private string lattitude = string.Empty;
        private string longitude = string.Empty;

        //variables for file writting a file
        StreamWriter file = null;
        private bool writeData = false;
                               
        //Create a serial port object to connect to in the event that a USB device is not present
        private SerialPort serial_ECGDevice = new SerialPort();
        private SerialPort serial_GPSDevice = new SerialPort();

        // This delegate enables asynchronous calls for setting the text property 
        //on a TextBox control. So we can call a GUI control from a different thread.
        delegate void SetTextCallback(double rate, double adc, double voltage, int[] filteredData, double freq);
        delegate void PrintLocationThread(string message);                
        
        //variable to hold the adc value
        short[] sECGData = new short[1];
        
        //variables to test input frequency of ECG device
        float freq = 0;
        /*float t1 = 0;
        float t2 = 0;
        int c = 0;*/
        #endregion

        #region ECGForm()
        public ECGForm()
        {
            InitializeComponent();
            ECGTraceView.BufferLength = 40 * 3;// 480 * 3;
            ECGTraceView.DataMax = 4096;
            ECGTraceView.DataMin = 0;
            ECGTraceView.FillBuffer(0);
            ECGTraceView.YMaxMeasure = 1500f / 1200f;
            ECGTraceView.YMinMeasure = -1500f / 1200f;
            ECGTraceView.SampleRate = 40;// 480;
            ConnectECGDevice();
        }
        #endregion

        #region ECGForm_Load
        private void ECGForm_Load(object sender, EventArgs e)
        {
            SetDefaults();
            SetParityValues(parityComboBox, GPSParity);
            SetStopBitValues(StopBitComboBox, GPSStopBits);
            SetPortNameValues(portComboBox, GPSPort);
        }
        #endregion

        #region ConnectECGDevice
        void ConnectECGDevice()
        {
            //This is the event that will create a USB interface.
            m_oECGDevice = ECGDevice.FindECGDevice(STM32_CIRCLE_ECG_VID, STM32_CIRCLE_ECG_VIP);            
            if (m_oECGDevice != null)	// did we find it?
            {
                MessageBox.Show("Connected to a USB Device!");
                // Yes! So wire into its events and update the UI
                m_oECGDevice.OnDeviceRemoved += new EventHandler(ECGDevice_OnDeviceRemoved);
                m_oECGDevice.OnInputReportDataReceived += new EKGReceiveDataEventHandler(ECGDevice_OnInputReportDataReceived);
            }
            //This is the event that I added to connect to a COM device when a USB device is not present
            else
            {
                //Didn't find USB device so connect a comport device                
                //This event will fire went there is data in the recieve buffer and call the ECGDevice_ReportDataReceived function.
                serial_ECGDevice.DataReceived += new SerialDataReceivedEventHandler(ECGDevice_ReportDataReceived);
                serial_GPSDevice.DataReceived += new SerialDataReceivedEventHandler(GPSDevice_ReportDataReceived);
            }
        }
        #endregion

        #region open port (Start Button)
        /// <summary>
        /// openBtn_Click will open a new port with the specified values.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void openBtn_Click(object sender, EventArgs e)
        {
            #region open ECG serial port
            try
            {
                serial_ECGDevice.PortName = portComboBox.Text;
                if (serial_ECGDevice.IsOpen)
                {
                    serial_ECGDevice.Close();
                }
                serial_ECGDevice.BaudRate = int.Parse(baudComboBox.Text);
                serial_ECGDevice.Parity = (Parity)Enum.Parse(typeof(Parity), parityComboBox.Text);
                serial_ECGDevice.DataBits = int.Parse(dataBitcomboBox.Text);
                serial_ECGDevice.StopBits = (StopBits)Enum.Parse(typeof(StopBits), StopBitComboBox.Text);
                serial_ECGDevice.ReceivedBytesThreshold = 24;                
                serial_ECGDevice.Open();
                serial_ECGDevice.DtrEnable = true;
                serial_ECGDevice.DiscardInBuffer();
            }
            catch
            {
                MessageBox.Show("No ECG Device Connected!");
            }
            #endregion

            #region open GPS serial port
            try
            {
                serial_GPSDevice.PortName = GPSPort.Text;
                if (serial_GPSDevice.IsOpen)
                {
                    serial_GPSDevice.Close();
                }
                serial_GPSDevice.BaudRate = int.Parse(GPSBuad.Text);
                serial_GPSDevice.Parity = (Parity)Enum.Parse(typeof(StopBits), GPSParity.Text);
                serial_GPSDevice.DataBits = int.Parse(GPSDataBits.Text);
                serial_GPSDevice.StopBits = (StopBits)Enum.Parse(typeof(StopBits), GPSStopBits.Text);
                serial_GPSDevice.Open();
                serial_GPSDevice.DtrEnable = true;
            }
            catch
            {
                MessageBox.Show("No GPS Device Connected!");
            }
            #endregion

            #region Control State
            openBtn.Enabled = false;
            closeBtn.Enabled = true;
            writeBtn.Enabled = true;
            stopWriteBtn.Enabled = false;
            portComboBox.Enabled = false;
            baudComboBox.Enabled = false;
            parityComboBox.Enabled = false;
            dataBitcomboBox.Enabled = false;
            StopBitComboBox.Enabled = false;
            GPSPort.Enabled = false;
            GPSBuad.Enabled = false;
            GPSDataBits.Enabled = false;
            GPSParity.Enabled = false;
            GPSStopBits.Enabled = false;
            #endregion
        }
        #endregion

        #region close port (Stop Button)
        /// <summary>
        /// closeBtn_Click will close the open COM port
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void closeBtn_Click(object sender, EventArgs e)
        {
            try
            {
                if (serial_ECGDevice.IsOpen)
                {
                    serial_ECGDevice.Close();
                }

                if (serial_GPSDevice.IsOpen)
                {
                    serial_GPSDevice.Close();
                }

                if (writeData)
                {
                    file.Close();
                }
            }
            catch { };


            //ECG control state
            openBtn.Enabled = true;
            closeBtn.Enabled = false;
            writeBtn.Enabled = false;
            portComboBox.Enabled = true;
            baudComboBox.Enabled = true;
            dataBitcomboBox.Enabled = true;
            parityComboBox.Enabled = true;
            StopBitComboBox.Enabled = true;

            //GPS control state
            GPSPort.Enabled = true;
            GPSBuad.Enabled = true;
            GPSDataBits.Enabled = true;
            GPSParity.Enabled = true;
            GPSStopBits.Enabled = true;

            //Record data control state
            stopWriteBtn.Enabled = false;
        }
        #endregion

        #region ECGDevice_OnDeviceRemoved
        /// <summary>
        /// ECG device has been removed : Called when read terminates with an exception on the background
        /// thread from the async read. Must use standard invokerequired/invoke practice.
        /// </summary>
        private void ECGDevice_OnDeviceRemoved(object sender, EventArgs e)
        {

            m_oECGDevice = null;
            if (InvokeRequired)
            {
                Invoke(new EventHandler(ECGDevice_OnDeviceRemoved), new object[] { sender, e });
            }
            else
            {
            }
        }
        #endregion

        #region ECGDevice_OnInputReportDataReceived
        /// <summary>
        /// ECG device data received : Called when read terminates normally on the background
        /// thread from the async read. Must use standard invokerequired/invoke practice.
        /// </summary>
        private void ECGDevice_OnInputReportDataReceived(object sender, EKGReceiveDataEventArgs args)
        {
            if (InvokeRequired)
            {
                try
                {
                    Invoke(new EKGReceiveDataEventHandler(ECGDevice_OnInputReportDataReceived), new object[] { sender, args });
                }
                catch
                {
                }
            }
            else
            {                
                ECGTraceView.AddDataAtEnd(filterProcess(args.data));
                ECGTraceView.Invalidate();
            }
        }
        #endregion

        #region ECGDevice_ReportDataReceived
        /// <summary>
        /// ECGDevice_ReportDataReceived is my function that reads data from the recieve buffer and plots
        /// it on the graphical display.
        /// </summary>
        private void ECGDevice_ReportDataReceived(object sender, SerialDataReceivedEventArgs args)
        {                        
            string moteData = string.Empty;
            byte[] ecgData = new byte[2];                       
            int[] filteredData = new int[39];
            double adc = 0;
            double voltage = 0;

            #region Find the frequency
            /*c++;
            if (c == 1)
            {
                t1 = DateTime.Now.Second + (float)DateTime.Now.Millisecond / 1000;                
            }
            if (c == 2)
            {
                t2 = DateTime.Now.Second + (float)DateTime.Now.Millisecond / 1000;
                float T = t2 - t1;                
                freq = 1 / T;                
                c = 0;
            }*/
            #endregion

            //Get the number of bytes in the buffer
            int bytes = serial_ECGDevice.BytesToRead;
            
            if (bytes > 23 )
            {
                byte[] packet = new byte[bytes];
                serial_ECGDevice.Read(packet, 0, bytes);
                serial_ECGDevice.DiscardInBuffer();
                //convert packet to a Hex string
                moteData = ByteToHex(packet);

                //get the data from the packet                       
                if (Convert.ToString(packet[0], 16) == "7e" && Convert.ToString(packet[1], 16) == "42")
                {
                    //packet is little endian
                    //channel 1
                    ecgData[0] = packet[20];
                    ecgData[1] = packet[19];
    
                    adc = Convert.ToDouble(ecgData[0]) * 256 + Convert.ToDouble(ecgData[1]);
                    voltage = 2.5 * adc / 4096;
                }

                #region write ecg data
                if (writeData && voltage > 0 && voltage < 2.5)
                {
                    WriteDataFile(voltage.ToString());
                }
                #endregion
               
                sECGData[0] = (short)adc;

                //convert byte to short
                //short[] sECGData = new short[ecgData.Length];
                //System.Buffer.BlockCopy(ecgData, 0, sECGData, 0, ecgData.Length);

                //filter and display the data
                if ((adc > 0) && (adc < 4096) && (voltage != 0))
                {
                    filteredData = filterProcess(sECGData);
                    CalculateHeartRate(filteredData, adc, voltage, freq);
                    ECGTraceView.AddDataAtEnd(filteredData);
                    ECGTraceView.Invalidate();
                }
            }
        }
        #endregion

        #region GPSDevice_ReportDataReceived
        /// <summary>
        /// GPSDevice_ReportDataReceived will read GPS data from a serial port
        /// </summary>
        /// <param name="sender">serial port object</param>
        /// <param name="args">data in the recieve buffer</param>
        private void GPSDevice_ReportDataReceived(object sender, SerialDataReceivedEventArgs args)
        {
            try
            {
                string msg = serial_GPSDevice.ReadLine();
                        
                //This Method was called from a back thread so we use this "if statement"
                //to call the GUI textboxes and update them with the new data.
                if (lattitudeTxtBx.InvokeRequired)
                {
                    PrintLocationThread g = new PrintLocationThread(PrintGPSLocation);
                    lattitudeTxtBx.Invoke(g, new object[] { msg });
                }
            }
            catch { }
        }
        #endregion

        #region filterProcess
        private int[] filterProcess(short[] data)
        {
            int i = 0;            
            int[] output = new int[data.Length];

            while (i < data.Length)
            {
                xv[0] = xv[1]; xv[1] = xv[2]; xv[2] = xv[3]; xv[3] = xv[4]; xv[4] = xv[5]; xv[5] = xv[6];
                xv[6] = (double)data[i] / gain;
                yv[0] = yv[1]; yv[1] = yv[2]; yv[2] = yv[3]; yv[3] = yv[4]; yv[4] = yv[5]; yv[5] = yv[6]; 
                yv[6] =   (xv[6] - xv[0]) + 3 * (xv[2] - xv[4])               
                 + ( -0.6175512499 * yv[0]) + (  3.9197050969 * yv[1])
                 + (-10.4691513780 * yv[2]) + ( 15.0642577980 * yv[3])
                 + (-12.3121230920 * yv[4]) + (  5.4148628120 * yv[5]);
                output[i++] = (short)(yv[6])+2048;
            }

            return output;
        }
        #endregion        
        
        #region CalculateHeartRate
        /// <summary>
        /// This method calculates and displays the heart rate of the patient
        /// </summary>
        private void CalculateHeartRate(int[] filteredData, double adc, double voltage, float freq)
        {                                                
            #region Create Square Wave from Input

            foreach (int data in filteredData)
            {                
                if (data > threshold)
                {                    
                    dataBuffer.Add(1);                    
                }
                else
                {
                    dataBuffer.Add(0);                    
                }
            }
            #endregion
            
            #region find Rwave and calculate heart rate
            //determine an rWave by a 0-1 change in the square wave            
            foreach (int data in dataBuffer)
            {                
                currentPt = data;
                if (currentPt == 0 && prevPt == 1)
                {
                    dataBuffer.Clear();
                    break;
                }
                else if (prevPt == 0 && currentPt == 1)
                {
                    rWave++; //rWave detected

                    endTime = DateTime.Now.Second + (double)DateTime.Now.Millisecond / 1000; //Time of second rWave detection
                    dataBuffer.Clear();

                    prevPt = currentPt;
                    period = endTime - startTime;
                    frequency = 1 / period;
                    heartRate = 60 * frequency;
                    startTime = endTime;

                    if (hrQ.Count == 10)
                    {
                        hrQ.Dequeue();
                        hrQ.Enqueue(heartRate);
                    }
                    else
                    {
                        hrQ.Enqueue(heartRate);
                    }

                    Array hrArray = hrQ.ToArray();
                    int hrArrayCount = hrArray.Length;
                    for (int i = 0; i < hrArrayCount; i++)
                    {
                        averageHeartRate = averageHeartRate + Convert.ToDouble(hrArray.GetValue(i));
                    }
                    averageHeartRate = averageHeartRate / hrArray.Length;
                    break;

                }
                prevPt = currentPt; //current point now becomes the previous point
            }
            prevPt = currentPt;
            #endregion
            
            adc = Convert.ToDouble(rWave);

            #region print heart rate in textbox
            //This Method was called from a back thread so we use this "if statement"
            //to call the GUI textboxes and update them with the new data.
            if (rateTxtBx.InvokeRequired)
            {
                SetTextCallback d = new SetTextCallback(Alarm);
                rateTxtBx.Invoke(d, new object[] { averageHeartRate, adc, voltage, filteredData, freq });
            }
            else
            {
                rateTxtBx.Text = heartRate.ToString();
            }
            #endregion
        }
        #endregion

        #region Alarm
        /// <summary>
        /// Called from the back thread to display heart rate and alarm messages
        /// if the heart rate is to low/high
        /// </summary>
        /// <param name="heartRate">float</param>
        private void Alarm(double heartRate, double adc, double voltage, int[] filteredData, double freq)
        {            
            /*i++;
            timeTxt.Text = i.ToString();
            adcTxt.Text = adc.ToString();
            voltageTxt.Text = voltage.ToString();            
            filteredDataTxt.Text = filteredData[0].ToString();
            freqTxtBx.Text = freq.ToString();
            */
            rateTxtBx.Text = Convert.ToInt16(heartRate).ToString();
            if (heartRate < 20)
            {
                alarmTxtBx.Text = "Low Heart Rate. Call 911!!!";
            }
            else if (heartRate > 200)
            {
                alarmTxtBx.Text = "High Heart Rate.";
            }
            else
            {
                alarmTxtBx.Text = "Normal No Problem.";
            }            
        }
        #endregion

        #region Print GPS Location
        /// <summary>
        /// Called from a seprate thread to write the lattitude and longitude to the GUI
        /// </summary>
        /// <param name="lat">lattitude</param>
        /// <param name="longg">longitude</param>
        private void PrintGPSLocation(string msg)
        {
            if (msg.Substring(0, 6) == "$GPGGA")
            {                
                lattitude = msg.Substring(18, 2) + "d " + msg.Substring(20, 7) + "'";
                if (msg.Substring(30, 1) == "0")
                {
                    longitude = msg.Substring(31, 2) + "d " + msg.Substring(33, 7) + "'";
                }
                else
                {
                    longitude = msg.Substring(30, 3) + "d " + msg.Substring(33, 7) + "'";
                }
                lattitudeTxtBx.Text = lattitude;
                longitudeTxtBx.Text = longitude;
                
                if (writeData)
                {
                    WriteDataFile(msg);
                }
            }
        }
        #endregion

        #region ByteToHex
        /// <summary>
        /// method to convert a byte array into a hex string
        /// </summary>
        /// <param name="comByte">byte array to convert</param>
        /// <returns>a hex string</returns>
        private string ByteToHex(byte[] comByte)
        {
            //create a new StringBuilder object
            StringBuilder builder = new StringBuilder(comByte.Length * 3);
            //loop through each byte in the array
            foreach (byte data in comByte)
                //convert the byte to a string and add to the stringbuilder
                builder.Append(Convert.ToString(data, 16).PadLeft(2, '0').PadRight(3, ' '));
            //return the converted value
            return builder.ToString().ToUpper();
        }
        #endregion

        #region SetDefaults
        /// <summary>
        /// SetDefaults is intialize the typical COM connection
        /// </summary>
        private void SetDefaults()
        {
            gainTxtBx.Text = gain.ToString();
            triggerTxtBx.Text = threshold.ToString();

            #region control state
            closeBtn.Enabled = false;
            writeBtn.Enabled = false;
            stopWriteBtn.Enabled = false;
            applyBtn.Enabled = false;
            #endregion            
            
            #region ECG defaults
            portComboBox.Text = "COM5";
            baudComboBox.SelectedText = "57600";
            dataBitcomboBox.SelectedIndex = 3;
            parityComboBox.Text = Parity.None.ToString();
            StopBitComboBox.Text = StopBits.One.ToString();
            #endregion

            #region GPS defaults
            GPSPort.Text = "COM7";
            GPSBuad.SelectedText = "38400";
            GPSDataBits.SelectedIndex = 3;
            GPSParity.Text = Parity.None.ToString();
            GPSStopBits.Text = StopBits.One.ToString();
            #endregion
        }
        #endregion

        #region SetPortNameValues
        /// <summary>
        /// SetPortNameValues will fill the combo box with the
        /// available COM ports.
        /// </summary>
        /// <param name="obj">Serial Port (COM)</param>
        public void SetPortNameValues(object obj1, object obj2)
        {

            foreach (string str in SerialPort.GetPortNames())
            {
                ((ComboBox)obj1).Items.Add(str);
                ((ComboBox)obj2).Items.Add(str);
            }
        }
        #endregion

        #region SetParityValues
        /// <summary>
        /// fills comboboxes with available parity values
        /// </summary>
        /// <param name="obj1"></param>
        /// <param name="obj2"></param>
        public void SetParityValues(object obj1, object obj2)
        {
            foreach (string str in Enum.GetNames(typeof(Parity)))
            {
                ((ComboBox)obj1).Items.Add(str);
                ((ComboBox)obj2).Items.Add(str);
            }
        }
        #endregion

        #region SetStopBitValues
        /// <summary>
        /// fills the combobox with available stop bit values
        /// </summary>
        /// <param name="obj1"></param>
        /// <param name="obj2"></param>
        public void SetStopBitValues(object obj1, object obj2)
        {
            foreach (string str in Enum.GetNames(typeof(StopBits)))
            {
                ((ComboBox)obj1).Items.Add(str);
                ((ComboBox)obj2).Items.Add(str);
            }
        }
        #endregion

        #region Record Data Button
        /// <summary>
        /// writeBtn_Click will write the GPS and ECG to a file
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void writeBtn_Click(object sender, EventArgs e)
        {
            file = new StreamWriter("data.txt");
            writeData = true;
            writeBtn.Enabled = false;
            stopWriteBtn.Enabled = true;
        }
        #endregion

        #region Stop Writing Data button
        private void stopWriteBtn_Click(object sender, EventArgs e)
        {
            if (writeData)
            {
                file.Close();
                writeData = false;
                writeBtn.Enabled = true;
                stopWriteBtn.Enabled = false;
            }
        }
        #endregion

        #region Write data to file
        /// <summary>
        /// write data to a file
        /// </summary>
        /// <param name="msg"></param>
        private void WriteDataFile(string msg)
        {
            file.WriteLine(msg);
        }
        #endregion

        #region Gain Text Box Changed
        private void gainTxtBx_TextChanged(object sender, EventArgs e)
        {            
            gainChanged = true;
            applyBtn.Enabled = true;
        }
        #endregion

        #region Trigger Text Box Changed
        private void triggerTxtBx_TextChanged(object sender, EventArgs e)
        {            
            trigger = true;
            applyBtn.Enabled = true;
        }
        #endregion

        #region Apply Button for Gain and Trigger
        private void applyBtn_Click(object sender, EventArgs e)
        {
            if (trigger)
            {
                threshold = Convert.ToInt32(triggerTxtBx.Text);
                trigger = false;
                applyBtn.Enabled = false;
            }

            if (gainChanged)
            {
                if (Convert.ToDouble(gainTxtBx.Text) < 5000 && Convert.ToInt16(gainTxtBx.Text) > 25)
                {
                    gain = Convert.ToDouble(gainTxtBx.Text);
                }
                else
                {
                    gainTxtBx.Text = "Gain out of bounds";
                }
                gainChanged = false;
                applyBtn.Enabled = false;
            }
        }
        #endregion        

        #region Exit
        /// <summary>
        /// btCancel_Click will close the COM port if it is open and 
        /// kill the GUI thread.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btCancel_Click(object sender, EventArgs e)
        {
            if (serial_ECGDevice.IsOpen)
            {
                serial_ECGDevice.Close();
            }
            
            if (serial_GPSDevice.IsOpen)
            {
                serial_GPSDevice.Close();
            }

            if (writeData)
            {
                file.Close();
            }            
            this.Close();
        }
        #endregion                                                                
    }
}